#!/bin/sh
#
# restores the file /var/lib/dpkg/status based on the
# files in /var/lib/dpkg/info and with the information of
# /var/lib/dpkg/available
#
# use the command line parameter -v to display verbose information
#
# NOTE: you do not have to, but you _SHOULD_ run this script as root, because
#       for calculating the md5sums of Conffiles you need to access all those
#       files, which a normal user usually is not allowed to.
#
# by Alexander Griesser <debian@tuxx-home.at>
# 2005-02-16
VERSION=1.0

# Display verbose information? (0 = off, 1 = on)
# may be overriden with the command line option "-v"
VERBOSE=0

CUR_INSTALLED=/tmp/cur_installed.txt
STATUS_NEW=/tmp/status.new
AVAILABLE=/var/lib/dpkg/available
STATUS=/var/lib/dpkg/status
INFO_DIR=/var/lib/dpkg/info

# path to awk (you'd better use gawk instead of mawk, if installed, because
# mawk limits the sprintf buffer to 1000 and something, which is way to few
# to get by with $CONFFILES)
AWK=/usr/bin/gawk

# Limits the number of lines grep will use for blockwise extracting
# of package information. Because of xlibs, this value has to be that
# high, xlibs has about 110 lines of status-information.
# If you are sure, that you can live with GREP_LIMIT=50, this script
# will run much faster, but 200 is the safe way.
GREP_LIMIT=200

# display usage information
function usage()
{
  echo "$(basename $0) - Usage information (Version: $VERSION)"
  echo
  echo "-v ... print verbose information"
  echo "-h ... print this screen"
  echo
  echo "This script was written by Alexander Griesser <debian@tuxx-home.at>"
  exit 0
}

# check command line parameters
if [ "$#" != "0" ]; then
  if [ "$#" = "1" ] && [ "$1" = "-v" ]; then
    VERBOSE=1
  else
    usage
  fi
fi

# check if the temporary files are still there
[ -e $CUR_INSTALLED ] && rm $CUR_INSTALLED
[ -e $STATUS_NEW ] && rm $STATUS_NEW

echo -n "Getting information about currently installed packages..."
cd /var/lib/dpkg/info && \
ls | sed 's/\(.*\)\.[^.]*$/\1/g' | sort -u >$CUR_INSTALLED

if [ "$?" = "0" ]; then
  echo "done."
else
  echo "failed!."
  exit 1
fi

echo -n "Extracting information about each package from $AVAILABLE..."
FIRST=1
while read line; do
  if [ "$VERBOSE" = "1" ]; then
    if [ "$FIRST" = "1" ]; then
      echo
      FIRST=0
    fi
    echo "Processing $line..."
  fi

  # check if we have to generate md5sums of configfiles
  if [ -e $INFO_DIR/$line.conffiles ]; then
    echo "Generating md5sums of conffiles..."
    CONFFILES="Conffiles:"
    while read conffile; do
      if [ "$VERBOSE" = "1" ]; then
        echo "Processing $conffile..."
      fi
      if [ -e "$conffile" ]; then
        CONFFILES="$CONFFILES\n $conffile $(md5sum $conffile | $AWK "-F " '{print $1}')"
      else
        if [ "$VERBOSE" = "1" ]; then
          echo "skipped (does not exist)."
        fi
      fi
    done <$INFO_DIR/$line.conffiles

    if [ "$VERBOSE" = "1" ]; then
      echo -e $CONFFILES
    fi

    grep -A $GREP_LIMIT "^Package: ${line}$" $AVAILABLE | grep -m 1 -B $GREP_LIMIT "^$" | grep -v "^\(Filename\|Size\|MD5sum\): " | $AWK '{if($0 == "Package: '$line'") {print $0"\nStatus: install ok installed"} else {print $0} }' | $AWK '{if($0 ~ /^Description: /) {print "'"$CONFFILES"'\n"$0 } else { print $0 }}' >>$STATUS_NEW
  else
    grep -A $GREP_LIMIT "^Package: ${line}$" $AVAILABLE | grep -m 1 -B $GREP_LIMIT "^$" | grep -v "^\(Filename\|Size\|MD5sum\): " | $AWK '{if($0 == "Package: '$line'") {print $0"\nStatus: install ok installed"} else {print $0} }' >>$STATUS_NEW
  fi
done <$CUR_INSTALLED
echo "done."
echo 
echo "==================================================="
echo "Your new status file has been saved to $STATUS_NEW."
echo "==================================================="

